﻿using System;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Interactivity;
using System.Windows.Interop;
using System.Windows.Media;
using GDIPlus = System.Drawing;

namespace Azyobuzi.Twikoto2.Behaviors
{
    class TransparentBackgroundBehavior : Behavior<Window>
    {
        protected override void OnAttached()
        {
            base.OnAttached();
            this.AssociatedObject.Loaded += this.AssociatedObject_Loaded;
        }

        protected override void OnDetaching()
        {
            this.AssociatedObject.Loaded -= this.AssociatedObject_Loaded;
            base.OnDetaching();
        }

        public Thickness Margins
        {
            get { return (Thickness)GetValue(MarginsProperty); }
            set { SetValue(MarginsProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Margins.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty MarginsProperty =
            DependencyProperty.Register("Margins", typeof(Thickness), typeof(TransparentBackgroundBehavior), new UIPropertyMetadata(new Thickness(0, 0, 0, 0)));

        private void AssociatedObject_Loaded(object sender, RoutedEventArgs e)
        {
            var w = sender as Window;
            w.Background = Brushes.Transparent;
            if (Environment.OSVersion.Version.Major >= 6 && DwmIsCompositionEnabled())
            {
                var source = HwndSource.FromVisual(w) as HwndSource;
                source.CompositionTarget.BackgroundColor = Color.FromArgb(0, 0, 0, 0);
                var desktop = GDIPlus.Graphics.FromHwnd(source.Handle);
                DwmExtendFrameIntoClientArea(source.Handle, new MARGINS()
                {
                    cxLeftWidth = Convert.ToInt32(this.Margins.Left * (desktop.DpiX / 96)),
                    cyTopHeight = Convert.ToInt32(this.Margins.Top * (desktop.DpiX / 96)),
                    cxRightWidth = Convert.ToInt32(this.Margins.Right * (desktop.DpiX / 96)),
                    cyBottomHeight = Convert.ToInt32(this.Margins.Bottom * (desktop.DpiX / 96))
                });
            }
            else
            {
                w.Background = SystemColors.GradientActiveCaptionBrush;
            }
        }

        [DllImport("dwmapi.dll", PreserveSig = false)]
        private static extern void DwmExtendFrameIntoClientArea(
            IntPtr hWnd, MARGINS pMargins);

        [DllImport("dwmapi.dll", PreserveSig = false)]
        public static extern bool DwmIsCompositionEnabled();
    }

    [StructLayout(LayoutKind.Sequential)]
    class MARGINS
    {
        public int cxLeftWidth;
        public int cxRightWidth;
        public int cyTopHeight;
        public int cyBottomHeight;
    }
}
